#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# mpr.py  -  Utilities for perfact::mpr
#
# Copyright (C) 2016 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

import tempfile
import os
import pytest  # for skipping tests
import six
import time

from .generic import safe_syscall, generate_random_string
# assert_safe_for_bash should probably go into .generic
from .ovpn import assert_safe_for_bash
from .file import tmp_cleanup, tmp_write

if not six.PY2:
    unicode = str


def mpr_getsshkey(keyloc='/home/zope/.ssh/mpr-id_rsa.pub'):
    """Checks if there is a ssh-key and returns it if there is one.
    Returns False if no ssh-key is found or raises an assertion error if the
    found pub-key file is faulty
    """
    if not os.path.isfile(keyloc):
        return False

    reader = open(keyloc, 'r')
    lines = reader.readlines()

    assert len(lines) == 1, 'Unexpected amount of lines in public key'

    return lines[0]


def mpr_sshkey(pkeyloc='/home/zope/.ssh/mpr-id_rsa',
               overwrite=False, keybitlen=4096):
    """Generate a new ssh-keypair if there isn't already one
    If there is one, only overwrite it when overwrite is True
    """
    # Return if there is a key and overwrite is False
    if os.path.isfile(pkeyloc) and not overwrite:
        return

    # Delete the current keys
    if os.path.isfile(pkeyloc):
        os.remove(pkeyloc)
    if os.path.isfile(pkeyloc+'.pub'):
        os.remove(pkeyloc+'.pub')

    gen_cmd = [
        '/usr/bin/ssh-keygen',
        # Set the bitlen
        '-b', str(keybitlen),
        # Set the keyname/location
        '-f', pkeyloc,
        # Prevent passphrases
        '-N', '']

    safe_syscall(gen_cmd, raisemode=True)


def mpr_makeconf(conf, signkey=None, digest='sha1'):
    """Creates a tar with a signed digest file for perfact::mpr >= 2.0

    Appends every file to a tar preserving owner, group and permissions.
    Creates a digest file containing checksums for every file.
    Signs the digest file with given key and digest algorithm and adds
    both files to the tar archive.
    Zip the tar and return it.

    Requirements are the 'tar' and 'openssl dgst' binaries.

    A tmpdir will be created in the process and the signkey has to be
    readable on disk.

    Input:
    A list of dictionaries containing file content and metadata e.g.
    * content
        conf=[{'path':'/example/file/path',
             'contents':u'some data: Öffentlich äußerst früh»¹²³',
             'unixuserid':10042,
             'unixgroupid':1023,
             'unixpermissions':1777},
             ]

    * signkey (optional)
    A path for private key for signing. The public key must exist on the
    perfact::mpr. If unset, no signature or digest are added to the
    archive.

    * digest
    A message digest algorithm which is supported by OpenSSL (dgst)
    The digest must be the same as expected by the perfact::mpr
    """
    tmpdir = tempfile.mkdtemp()

    # Create an empty tar-archive of the form /tmp/xyz/xyz.tar and append files
    tar = tmpdir+'/'+tmpdir.split('/')[-1]+'.tar'
    cmd = '/bin/tar -cf '+tar+' --files-from /dev/null'
    ret, out = safe_syscall(cmd)
    assert ret == 0, 'Tarfile %s could not be created!\n' \
        'cmd: %s\n' \
        'out: %s\n' \
        'ret: %s' % (tar, cmd, out, ret)

    for f in conf:
        # setup the environment, parse the delivered 'conf'
        pathlist = f['path'].split('/')
        dirs = '/'.join(pathlist[:-1])
        filename = pathlist[-1]
        filepath = tmpdir+dirs+'/'+filename

        # use sane defaults if nothing is configured
        uid = str(f['unixuserid'] or '0')
        gid = str(f['unixgroupid'] or '0')
        perms = str(f['unixpermissions'] or '644')
        contents = f['contents']

        # convert Unicode to UTF-8 for linux files
        if isinstance(contents, unicode):
            contents = contents.encode('utf-8')

        # write file to disk in directory structure
        if not os.path.isdir(tmpdir+dirs):
            os.makedirs(tmpdir+dirs)
        fd = open(filepath, 'wb')
        fd.write(contents)
        fd.flush()
        fd.close()

        # Set the unix-filepermissions e.g. rw-r--r--
        cmd = '/bin/chmod '+perms+' '+filepath
        ret, out = safe_syscall(cmd)
        assert ret == 0, 'Could not set file permissions to: %s for %s!\n' \
            'Check mount options for /tmp filesystem!\n' \
            'cmd: %s\n' \
            'out: %s\n' \
            'ret: %s' % (perms, filepath, cmd, out, ret)

        # Calculate the hashsum of the file
        # this gives us something like: SHA1= a0b1c2d3e423f891
        dgstval = openssl_digest(contents=contents, digest=digest)

        # the form has to be: SHA1(/path/to/file)= a0b1c2d3e423f891
        dgstval = dgstval.replace('=', '('+dirs+'/'+filename+')=')
        dgstval = dgstval+'\n'

        # append the new filehash to the signature file
        fd = open(tmpdir+'/mpafiles.dgst', 'a')
        fd.write(dgstval)
        fd.flush()
        fd.close()

        # Add the current file to the tar with correct owner and group
        cmd = '/bin/tar -C '+tmpdir+' -rf '+tar+' \
        --numeric-owner --owner='+uid+' --group='+gid+' -p .'+dirs+'/'+filename
        ret, out = safe_syscall(cmd)
        assert ret == 0, 'File: %s could not be added to the tar: %s!\n' \
            'cmd: %s\n' \
            'out: %s\n' \
            'ret: %s' % (dirs+'/'+filename, tar, cmd, out, ret)

    if signkey is not None:
        # digitally sign the digest file
        fd = open(tmpdir+'/mpafiles.dgst', 'r')
        dgstfile = fd.read()
        fd.close()

        signature = openssl_sign(contents=dgstfile, digest=digest, key=signkey)
        fd = open(tmpdir+'/mpafiles.dgst.signature', 'wb')
        fd.write(signature)
        fd.close()

        # add the digest file to the tar
        cmd = '/bin/tar -C '+tmpdir+' -rf '+tar+' mpafiles.dgst'
        ret, out = safe_syscall(cmd)
        assert ret == 0, 'Digest file could not be added to the tar!\n' \
            'cmd: %s\n' \
            'out: %s\n' \
            'ret: %s' % (cmd, out, ret)

        # add the signature file of the digest file at last
        cmd = '/bin/tar -C '+tmpdir+' -rf '+tar+' mpafiles.dgst.signature'
        ret, out = safe_syscall(cmd)
        assert ret == 0, 'Signature could not be added to the tar!\n' \
            'cmd: %s\n' \
            'out: %s\n' \
            'ret: %s' % (cmd, out, ret)

    # gzip the tar
    cmd = '/bin/gzip '+tar
    ret, out = safe_syscall(cmd)
    assert ret == 0, 'Could not zip tarfile %s!\n' \
        'cmd: %s\n' \
        'out: %s\n' \
        'ret: %s' % (tar, cmd, out, ret)

    # return the zipped content of the archive
    tgz = tmpdir+'/'+tmpdir.split('/')[-1]+'.tar.gz'
    fd = open(tgz, 'rb')
    tgz = fd.read()
    fd.close()

    # clean up
    for dirpath, dirnames, filenames in os.walk(tmpdir, topdown=False):
        for name in filenames:
            os.remove(os.path.join(dirpath, name))
        for name in dirnames:
            os.rmdir(os.path.join(dirpath, name))
    os.rmdir(tmpdir)

    return tgz


def openssl_sign(contents, key, digest='sha1'):
    """Digitally sign the contents with the given algorithm and key.
    Inputs are strings (unicode).
    Returns binary data (bytes).
    """
    tmpcontents = tempfile.mktemp()
    fd = open(tmpcontents, 'w')
    fd.write(contents)
    fd.close()

    tmpkey = tempfile.mktemp()
    fd = open(tmpkey, 'w')
    fd.write(key)
    fd.close()

    cmd = 'openssl dgst -'+digest+' -sign '+tmpkey+' '+tmpcontents
    ret, out = safe_syscall(cmd, text=False)
    assert ret == 0, 'Signature could not be created for: \n\
        %s with key: %s \n\
        cmd: %s \n\
        error: %s \n\
        returncode: %s ' % (tmpcontents, tmpkey, cmd, out, ret)
    os.remove(tmpcontents)
    os.remove(tmpkey)
    return out


def openssl_digest(contents, digest='sha1'):
    """Calculate digest (checksum) for input with the given algorithm
    using 'openssl dgst'
    Inputs are bytes, normally UTF-8 encoded data.

    Returns string containing the digest algorithm and digest e.g.
        SHA1= a35bcb9de70212c9594b77e6b3bcaf7897fe3e02
    """
    tmp = tempfile.mktemp()
    fd = open(tmp, 'wb')
    fd.write(contents)
    fd.close()

    cmd = '/usr/bin/openssl dgst -'+digest+' '+tmp
    ret, out = safe_syscall(cmd)
    assert ret == 0, 'Digest could not be calculated for: %s' % tmp
    out = out.replace('('+tmp+')', '')
    out = out.strip()
    os.remove(tmp)
    return out


def sshRunCmd(ip, cmd, user='maint', parameter=None,
              identity='/home/zope/.ssh/mpr-id_rsa', dryrun=False,
              proxyhost=None, proxyuser=None):
    """Send a command, which has to be in the allowed commands list,
    to the specified ip-address via ssh and return the results.
    All inputs, except 'cmd' are checked for bash-magic.

    :para ip: IP address or hostname
    :param cmd: A list containing the commands to be executed on the MPR
                (first item is the command which must exist in whitelist!)
    :param user: User part for the SSH call e.g. 'user@host'
    :param parameter: Optional string containing parameters for the SSH call
    :param proxyhost: The hostname of a server acting as a proxy. A SSH
                      connection will be established to this server first
                      and then all subsequent commands are run.
    :param proxyuser: The username to use for the SSH connection to the
                      given proxyhost

    It is intended to use this function for connections to an MPR as user
    'zope' - hence the usage of the special identity / key file.

    This function does not handle any errors other than 'command not allowed'.

    >>> sshRunCmd(ip='8.9.10.11', cmd=['/bin/ping'], parameter='-2',
    ...     dryrun=True)  # doctest: +SKIP
    ['ssh', '-o', 'StrictHostKeyChecking=no', '-i', \
'/home/zope/.ssh/mpr-id_rsa', '-2', 'maint@8.9.10.11', '/bin/ping']

    >>> sshRunCmd(ip='127.0.0.1', cmd=['ping', '-c1', '-n'], identity='/',
    ...     dryrun=True)  # doctest: +SKIP
    ['ssh', '-o', 'StrictHostKeyChecking=no', '-i', '/', 'maint@127.0.0.1', \
'ping -c1 -n']
    """
    proxy_cmd = []
    if proxyhost and proxyuser:
        usert_to_host = '{proxyuser}@{proxyhost}'.format(
            proxyuser=proxyuser,
            proxyhost=proxyhost
        )
        proxy_cmd = ['ssh', '-o', 'StrictHostKeyChecking=no', usert_to_host]

    allowedcmds = [
        'ping',
        '/bin/ping',
    ]

    prog = cmd[0]
    args = None
    if len(cmd) > 1:
        args = cmd[1:]
        cmd = str(prog) + ' ' + ' '.join(args)
    else:
        cmd = prog

    if prog not in allowedcmds:
        raise AssertionError("Command is not allowed!\n\
        Rejected: %s" % str(cmd))

    # special handling for user input - take care of any bash magic!
    inputs = [a for a in [ip, cmd, user, parameter, identity, args]
              if a is not None]
    for user_input in inputs:
        assert_safe_for_bash(user_input)

    # build a command list to be executed on this server
    # connect to any host without questions
    local_cmd = proxy_cmd + ['ssh', '-o', 'StrictHostKeyChecking=no']

    # check path to identity file - if any
    if identity is not None:
        assert os.path.exists(identity), \
            'Identity / key file %s does not exist' % identity
        local_cmd.extend(['-i', identity])

    # add optional parameters to the SSH call
    if parameter is not None:
        local_cmd.append(parameter)

    # add the 'user@host' part
    local_cmd.append(str(user)+'@'+str(ip))

    # finally add the command which should run on the host
    local_cmd.append(cmd)

    # for testing purposes
    if dryrun:
        return (local_cmd)

    # run it on this host
    ret, out = safe_syscall(local_cmd)
    return (ret, out)


def mpr_scp(ip, file, location, proxyhost, proxyuser, identity, user='maint',
            chmod='', chown='', conntimeout=10, dryrun=False):
    '''
    Copies a file to a desired location onto an MPR.

    IMPORTANT: The MPR user supplied by 'user' (default: 'maint') must be
               allowed to execute commands as root without being asked
               for a password. This is the default on newer MPRs.

    Command chain:
        scp -> scp -> mv -> chown -> chmod
        Note: File will be first copied to the proxy and then to the mpr
        Note: The proxy can be the current server too

    Return codes:
        1: Got directory instead of full path
        2: Input not safe for bash
        3: scp command to proxy failed
        4: scp command to mpr failed
        5: mv command failed
        6: chown command failed
        7: chmod command failed

    :param ip: The target ip of the desired MPR as string
    :param file: The content of a file as bytes
    :param location: The full path where the file should be copied
                     to on the MPR as string
    :param user: User who connects to the MPR as string. Default: maint
    :param chmod: Set filemode as string.
    :param chown: Set file owner and group as string.
    :param identity: The ssh key to identify ourselves to the MPR
    :param conntimeout: When to timeout scp/ssh commands in seconds as int
    :param dryrun: Dryrun set to True will return the commands which
                   would have been executed
    :param proxyhost: The hostname of a server acting as a proxy. A SSH
                      connection will be established to this server first
                      and then all subsequent commands are run.
    :param proxyuser: The username to use for the SSH connection to the
                      given proxyhost
    :returns: A dict with the function specific return code as int, the
              specific error code from an action which failed as int and
              a message which describes the potential error.
              {'status': <status>, 'error': <error>, 'msg': <message>}
    '''
    # Dictionary which will be returned
    return_data = {'status': 0, 'error': 0, 'cmd': '', 'msg': 'Success'}

    # The location must be a full path including the filename
    if location.endswith('/'):
        return_data['status'] = 1
        return_data['msg'] = (
            'Please specify the full path and not just the directory'
        )
        return return_data

    # Assure bash safety
    try:
        assert_safe_for_bash(ip)
        assert_safe_for_bash(location)
        assert_safe_for_bash(user)
        assert_safe_for_bash(chmod)
        assert_safe_for_bash(chown)
        assert_safe_for_bash(identity)
        assert_safe_for_bash(proxyhost)
        assert_safe_for_bash(proxyuser)
    except AssertionError as err:
        return_data['status'] = 2
        return_data['msg'] = unicode(err)
        return return_data

    # Default all commands which potentially will be build
    cmd_ssh_proxy = cmd_cp_proxy = cmd_cp_mpr = cmd_mv_mpr = \
        cmd_chown_mpr = cmd_chmod_mpr = ''

    # Generate a path for the local server, the proxy and the MPR
    # The proxy can be the current server too, that's why we need different
    # names for the tmp file
    if dryrun:
        local_file = '/tmp/dryrun_file'
        proxy_file = '/tmp/dryrun_file2'
        mpr_file = '/tmp/dryrun_file3'
    else:
        local_file = os.path.join(
            '/tmp/',
            generate_random_string(length=4) + unicode(time.time())
        )
        proxy_file = os.path.join(
            '/tmp/',
            generate_random_string(length=4) + unicode(time.time())
        )
        mpr_file = os.path.join(
            '/tmp/',
            generate_random_string(length=4) + unicode(time.time())
        )
        # Create the file on the local server
        tmp_write(local_file, file)

    # Command to SSH to the proxy
    # Commands will run in conjunction with this one
    cmd_ssh_proxy = (
        'ssh -o StrictHostKeyChecking=no -o ConnectTimeout={conntimeout} '
        '{proxyuser}@{proxyhost} '
        ).format(
        conntimeout=conntimeout,
        proxyuser=proxyuser,
        proxyhost=proxyhost
    )
    # Command to copy a file to the proxy
    cmd_cp_proxy = (
        'scp {local_file} '
        '{proxyuser}@{proxyhost}:{proxy_file}'
    ).format(
        local_file=local_file,
        proxy_file=proxy_file,
        proxyuser=proxyuser,
        proxyhost=proxyhost
    )

    # Command to copy the file from the proxy to the MPR
    cmd_cp_mpr = cmd_ssh_proxy + (
        'scp -o StrictHostKeyChecking=no -o ConnectTimeout={conntimeout} '
        '-i {identity} {file} {user}@{ip}:{location}'
    ).format(
        conntimeout=conntimeout,
        identity=identity,
        file=proxy_file,
        user=user,
        ip=ip,
        # Copy file to /tmp/ to move it later as sudo
        location=mpr_file
    )

    # Build the generic ssh command to attach and execute other commands
    # on the MPR.
    cmd_ssh_mpr = cmd_ssh_proxy + (
        'ssh -o StrictHostKeyChecking=no -o ConnectTimeout={conntimeout} '
        '-i {identity} {user}@{ip} '
    ).format(
        conntimeout=conntimeout,
        identity=identity,
        user=user,
        ip=ip
    )

    # Build command to move the file on the MPR from /tmp/ to the desired
    # location.
    cmd_mv_mpr = cmd_ssh_mpr + 'sudo mv {file} {location}'.format(
        # Location of file is the same on the Proxy and MPR
        file=mpr_file,
        location=location
    )
    # If we want to chown the file
    if chown:
        cmd_chown_mpr = (
                cmd_ssh_mpr + 'sudo chown {chown}:{chown} {file}'
        ).format(
            chown=chown,
            file=location
        )
    # If we want to chmod the file
    if chmod:
        cmd_chmod_mpr = cmd_ssh_mpr + 'sudo chmod {chmod} {file}'.format(
            chmod=chmod,
            file=location
        )

    # Return created commands for testing
    if dryrun:
        # Dryrun returns a tuple of used commands
        commands = (
            cmd_cp_proxy, cmd_cp_mpr, cmd_mv_mpr, cmd_chown_mpr,
            cmd_chmod_mpr
        )
        commands = '\n'.join((cmd for cmd in commands if cmd))
        return_data['msg'] = commands
        return return_data

    # Copy file to the proxy
    ret, out = safe_syscall(cmd_cp_proxy, raisemode=False)
    if ret:
        return_data['status'] = 3
        return_data['error'] = ret
        return_data['msg'] = out
        return_data['cmd'] = cmd_cp_proxy
        return return_data

    # Execute the scp command, this command will always be executed
    # This will copy the file to the MPR from the given proxy server
    ret, out = safe_syscall(cmd_cp_mpr, raisemode=False)

    # Delete the tmp file on the mpa server
    tmp_cleanup(local_file)
    # Delte the tmp file on the proxy
    rm_file = cmd_ssh_proxy + 'rm {file_path}'.format(
        file_path=proxy_file
    )
    safe_syscall(rm_file, raisemode=False)

    # Check return code of previous copy command
    if ret:
        return_data['status'] = 4
        return_data['error'] = ret
        return_data['msg'] = out
        return_data['cmd'] = cmd_cp_mpr
        return return_data

    # Setup the file we copied to the MPR (mv, chown, chmod)
    ret, out = safe_syscall(cmd_mv_mpr, raisemode=False)
    if ret:
        return_data['status'] = 5
        return_data['error'] = ret
        return_data['msg'] = out
        return_data['cmd'] = cmd_mv_mpr
        return return_data

    if chown:
        ret, out = safe_syscall(cmd_chown_mpr, raisemode=False)
        if ret:
            return_data['status'] = 6
            return_data['error'] = ret
            return_data['msg'] = out
            return_data['cmd'] = cmd_chown_mpr
            return return_data

    if chmod:
        ret, out = safe_syscall(cmd_chmod_mpr, raisemode=False)
        if ret:
            return_data['status'] = 7
            return_data['error'] = ret
            return_data['msg'] = out
            return_data['cmd'] = cmd_chmod_mpr
            return return_data

    # Delte the tmp file on the MPR
    rm_file = cmd_ssh_mpr + 'sudo rm {file_path}'.format(
        file_path=mpr_file
    )
    safe_syscall(rm_file, raisemode=False)

    return return_data


def mpr_applyrules(ip, proxyhost, proxyuser, identity, user='maint',
                   down_script='down-cmd-tun-new', conntimeout=10,
                   dryrun=False):
    '''
    This function will update the rules on the MPR.

    Sequence:
    1. Execute old down-cmd-tun script
    2. Execute up-cmd-tun script
    3. Move down-cmd-tun-new to down-cmd-tun (Replace old one with new one)

    Return codes:
        1: Input not safe for bash
        2: Command to apply the new rules failed

    :param ip: The target ip of the desired MPR as string
    :param user: User who connects to the MPR as string. Default: maint
    :param down_script: The name of the down script which is already placed
                        onto the MPR under /etc/openvpn/<down_script> as string
    :param identity: The ssh key to identify ourselfs to the MPR
    :param conntimeout: When to timeout ssh commands in seconds as int
    :param dryrun: Dryrun set to True will return the command which
                   would have been executed
    :param proxyhost: The hostname of a server acting as a proxy. A SSH
                      connection will be established to this server first
                      and then all subsequent commands are run.
    :param proxyuser: The username to use for the SSH connection to the
                      given proxyhost
    :returns: A dict with the function specific return code as int, the
              specific error code from an action which failed as int and
              a message which describes the potential error.
              {'status': <status>, 'error': <error>, 'msg': <message>}
    '''
    # Dictionary which will be returned
    return_data = {'status': 0, 'error': 0, 'cmd': '', 'msg': 'Success'}

    # Assure bash safety
    try:
        assert_safe_for_bash(ip)
        assert_safe_for_bash(user)
        assert_safe_for_bash(down_script)
        assert_safe_for_bash(identity)
    except AssertionError as err:
        return_data['status'] = 1
        return_data['msg'] = unicode(err)
        return return_data

    # Path of the new down script currently placed on the mpr
    new_down_script = os.path.join('/etc/openvpn/', down_script)

    # We will acess the MPR from our proxy
    cmd = (
        'ssh -o StrictHostKeyChecking=no -o ConnectTimeout={conntimeout} '
        '{proxyuser}@{proxyhost} '
        ).format(
        conntimeout=conntimeout,
        proxyuser=proxyuser,
        proxyhost=proxyhost
    )

    # Command to connect to the mpr and apply the rules
    cmd += (
        'ssh -o StrictHostKeyChecking=no -o ConnectTimeout={conntimeout} '
        '-i {identity} {user}@{ip} '
        '"sudo bash /etc/openvpn/down-cmd-tun '
        '&& sudo bash /etc/openvpn/up-cmd-tun '
        '&& sudo mv {new_down_script} /etc/openvpn/down-cmd-tun"'
    ).format(
        conntimeout=conntimeout,
        identity=identity,
        user=user,
        ip=ip,
        new_down_script=new_down_script,
    )

    if dryrun:
        return_data['msg'] = cmd
        return return_data

    ret, out = safe_syscall(cmd, raisemode=False)
    if ret:
        return_data['status'] = 2
        return_data['error'] = ret
        return_data['msg'] = out
        return_data['cmd'] = cmd
        return return_data

    return return_data


def mpa_signkey(expire_days=None, passphrase=None,
                mpasign_path='/home/zope/mpasign'):
    '''Generate a key for signing MPA software updates. '''
    from .cert import cert_makekey, cert_getpub

    key = cert_makekey(passphrase=passphrase)
    pub = cert_getpub(key=key, passphrase=passphrase)

    safe_syscall(['mkdir', '-p', '-m', '700', mpasign_path], raisemode=False)

    fh = open(mpasign_path+'/mpasignature.pub', 'w')
    fh.write(pub)
    fh.close()

    fh = open(mpasign_path+'/mpasignkey.pem', 'w')
    fh.write(key)
    fh.close()

    return key, pub


def mpa_getsignpub(mpasign_path='/home/zope/mpasign'):
    '''Export the public key for the perfact::mpr to verify signatures'''
    fh = open(mpasign_path+'/mpasignature.pub', 'r')
    pub = fh.read()
    fh.close()

    return pub


def mpa_getsignkey(mpasign_path='/home/zope/mpasign'):
    fh = open(mpasign_path+'/mpasignkey.pem', 'r')
    key = fh.read()
    fh.close()

    return key


# The following code is for compatibility with perfact::mpr < 2.0
# !! Highly deprecated and untested !!
# Configfiles are no longer encrypted using the public-key!
# This is needed for compatibility with old perfact::mprs
include_deprecated_code = True

if include_deprecated_code:

    capath = '/home/zope/mpasign/legacy'

    def tar_to_conf(tarfile):
        """WARNING: This code is deprecated and should only be used
        to support legacy devices.

        Extract config files from a tar archive of perfact::mpr 1.0
        """
        res = []
        tmpdir = tempfile.mkdtemp()
        fd = open(tmpdir+'.tgz', 'wb')
        fd.write(tarfile)
        fd.close()
        os.system('cd '+tmpdir+'; tar zxf '+tmpdir+'.tgz')
        for root, dirs, files in os.walk(tmpdir):
            for name in files:
                fd = open(os.path.join(root, name))
                contents = fd.read()
                fd.close()
                res.append({
                    'path': os.path.join(root, name).replace(tmpdir, ''),
                    'contents': contents,
                })
        os.system('rm '+tmpdir+'.tgz')
        os.system('rm -rf '+tmpdir)
        return res

    def conf_to_tar(conf):
        """WARNING: This code is deprecated and should only be used
        to support legacy devices.

        Create a tar containing files for perfact::mpr 1.0
        """
        tmpdir = tempfile.mkdtemp()
        for f in conf:
            c = f['path'].split('/')
            filename = c[-1]
            dir = '/'.join(c[:-1])
            try:
                os.makedirs(tmpdir+dir)
            except OSError:
                pass
            contents = f['contents']
            fd = open(tmpdir+dir+'/'+filename, 'w')
            fd.write(contents)
            fd.flush()
            fd.close()
        os.system('cd '+tmpdir+'; tar zcf '+tmpdir+'.tgz *')
        fd = open(tmpdir+'.tgz', 'rb')
        res = fd.read()
        fd.close()
        os.system('rm -rf '+tmpdir)
        os.system('rm '+tmpdir+'.tgz')
        return res

    def encryptConf(conf):
        """WARNING: This code is deprecated and should only be used
        to support legacy devices.

        Encrypt a file (presumably tar.gz from perfact::mpr) with the
        perfact::mpr rollout key
        """
        # expects a single .tgz file to encrypt it using the public
        # mprrollout.pem
        tmpdir = tempfile.mkdtemp()
        fd = open(tmpdir+'/conf.tgz', 'wb')
        fd.write(conf)
        fd.close()

        ret, out = safe_syscall(
            '/usr/bin/openssl smime -encrypt -in ' + tmpdir +
            '/conf.tgz -out ' + tmpdir + '/conf.crypt -binary ' +
            '-noattr -aes256 ' + capath + '/certs/mprrollout.pem'
        )
        if ret != 0:
            raise AssertionError('Encrypting the files with openssl failed. '
                                 'Errorlevel: %s Message: %s' % (ret, out))

        fd = open(tmpdir+'/conf.crypt', 'rb')
        crypt = fd.read()
        fd.close()
        # cleanup
        os.system('rm -rf '+tmpdir)
        return crypt

    def wrapTar(config=None, certs=None):
        """WARNING: This code is deprecated and should only be used
        to support legacy devices.

        Create a tar.gz archive from input files.
        """
        tmpdir = tempfile.mkdtemp()
        if config:
            fd = open(tmpdir+'/config.tgz', 'wb')
            fd.write(config)
            fd.flush()
            fd.close()
        if certs:
            fd = open(tmpdir+'/certs.tgz', 'wb')
            fd.write(certs)
            fd.flush()
            fd.close()
        # enclose all files in the tmpdir into a single tar-archive
        os.system('cd '+tmpdir+'; tar cf '+tmpdir+'.tar *')
        fd = open(tmpdir+'.tar', 'rb')
        res = fd.read()
        fd.close()
        # cleanup
        os.system('rm -rf '+tmpdir)
        # os.system('rm '+tmpdir+'.tar')
        return res

    def prepareMPAinfo(mpaurl, commid, certid=None, p12=None, cacert=None):
        """WARNING: This code is deprecated and should only be used
        to support legacy devices.

        Create tar.gz containing the perfact::mpr identity information
        and MPA server endpoint URL.
        """
        # Create tmp directory from which the archive will be crated
        tmpdir = tempfile.mkdtemp()

        # If we got the certid, we will generate a p12 archive out of it
        if certid is not None:
            try:
                passphr = 'MPrRoLlOuT'
                # WARNING: zocalib does not exist anymore on newer systems
                z = zocalib.ZoCA()
                p12 = z.loadCert(certid, format='p12', passphr=passphr)
            except NameError:
                # Cleanup tmp directory
                os.system('rm -rf ' + tmpdir)
                raise AssertionError("Legacy dependency 'zocalib' not found")

        # Write the p12 archive inside the tmp directory
        if p12 is not None:
            with open(tmpdir + '/cert.p12', 'wb') as f:
                f.write(p12)

        # Write the commid inside the tmp directory
        with open(tmpdir + '/commid', 'w') as f:
            f.write(commid)

        # Write the mpaurl inside the tmp directory
        with open(tmpdir + '/mpaurl', 'w') as f:
            f.write(mpaurl)

        if cacert is not None:
            # If the cacert was given as an argument
            # we create the file inside the tmp dir
            with open(tmpdir + '/cacert.pem', 'w') as f:
                f.write(cacert)
        else:
            # If not we copy the cacert fromt he hardrive to the tmp directory
            os.system(
                'cp ' + capath + '/certs/cacert.pem ' + tmpdir + '/cacert.pem'
            )

        # Create an archive out of the stored files inside the tmp directory
        os.system('cd ' + tmpdir + '; tar czf certs.tgz *')

        # Copy the archive by reading it bytes
        archive = open(tmpdir + '/certs.tgz', 'rb')
        archive_bytes = archive.read()
        archive.close()

        # cleanup tmp directory
        os.system('rm -rf ' + tmpdir)

        return archive_bytes

    def filename_search_ext(term, base='/opt/mpr/1.0'):
        """WARNING: This code is deprecated and should only be used
        to support legacy devices.

        Generates a list with a maximum of 10 paths starting from the
        given base path.
        """
        if term.rfind('/') > 0:
            if term.endswith('/'):
                sbase = term[:-1]
                sterm = ''
            else:
                sbase = term[:term.rfind('/')]
                sterm = term[term.rfind('/')+1:]
        else:
            sbase = ''
            sterm = term
        fsbase = base+sbase
        cand = os.listdir(fsbase)
        if sterm:
            cand = [a for a in cand if a.startswith(sterm)]
        res = list([sbase+'/'+a for a in cand])
        res.sort()
        return res[:10]

    def mpr_getfile(path, base='/opt/mpr/1.0'):
        """WARNING: This code is deprecated and should only be used
        to support legacy devices.

        Return the content of a file given in the path argument.
        """
        if path and path.find('..') > 0:
            return ''
        if path:
            try:
                f = open(base+path, 'rb')
                contents = f.read()
                f.close()
                return contents
            except OSError:
                pass
        return ''

# End of deprecated code


@pytest.mark.skip(reason="Uses assumptions on the current user")
def test_all():
    """Test sequence of new functions (MPR >= 2.0)"""
    from .fileassets import fileassets

    import os
    import pwd

    def file_owner(filename):
        return pwd.getpwuid(os.stat(filename).st_uid).pw_name

    # calculate checksum
    expected = 'SHA1= 4b2c0a18ee80fe3f72f7878608dbd13101af8010'
    contents = fileassets['tests.mpr_test']
    chksum = openssl_digest(contents=contents, digest='sha1')
    assert expected == chksum, (
        'Checksums do not match! expected:'
        ' %s got: %s' % (expected, chksum)
    )

    # load private key
    key = fileassets['tests.mpr_rsa_key']

    # sign and verify signature
    expected = fileassets['tests.mpr_testsignature']

    signature = openssl_sign(contents=contents, key=key, digest='sha1')
    assert expected == signature, 'Signatures do not match!'

    # testdata
    conf = [
        {
            'path': '/example/path/file1',
            'contents': u'some data: Öffentlich äußerst früh»¹²³',
            'unixuserid': 10042,
            'unixgroupid': 10023,
            'unixpermissions': 777
        },
        {
            'path': '/example/path/file2',
            'contents': u'some data: Öffentlich äußerst früh»¹²³',
            'unixuserid': 10043,
            'unixgroupid': 10024,
            'unixpermissions': 1437
        },
        {
            'path': '/example/path/file3',
            'contents': u'some data: Öffentlich äußerst früh»¹²³',
            'unixuserid': None,
            'unixgroupid': None,
            'unixpermissions': None
        },
    ]

    # Make a mpr config
    tgz = mpr_makeconf(conf=conf, signkey=key, digest='sha1')

    # unpack mpr config
    tmpdir = tempfile.mkdtemp()
    fd = open(tmpdir+'/config.tgz', 'wb')
    fd.write(tgz)
    fd.close()

    cmd = '/bin/tar -C '+tmpdir+' -p -zxf '+tmpdir+'/config.tgz'
    ret, out = safe_syscall(cmd)
    assert ret == 0, 'Could not unpack tgz! %s' % out

    cmd = '/bin/tar -C '+tmpdir+' -p -tvzf '+tmpdir+'/config.tgz'
    ret, out = safe_syscall(cmd)
    assert ret == 0, 'Could not test tgz! %s' % out

    import re
    target_output = '''\
-rwxrwxrwx 10042/10023      46 ....-..-.. ..:.. ./example/path/file1
-r---wxrwt 10043/10024      46 ....-..-.. ..:.. ./example/path/file2
-rw-r--r-- 0/0              46 ....-..-.. ..:.. ./example/path/file3
-rw-r--r-- perfact/perfact 204 ....-..-.. ..:.. mpafiles.dgst
-rw-r--r-- perfact/perfact 128 ....-..-.. ..:.. mpafiles.dgst.signature\
'''
    assert re.match(target_output, out), (
        'tgz contents test failed. result:\n%s expected:\n%s' %
        (out, target_output)
    )

    # load public key
    pub = fileassets['tests.mpr_rsa_pub']
    fd = open(tmpdir+'/mpasignature.pub', 'w')
    fd.write(pub)
    fd.close()

    # verify signature
    cmd = (
        'openssl dgst -sha1 -verify ' + tmpdir +
        '/mpasignature.pub -signature ' + tmpdir +
        '/mpafiles.dgst.signature ' + tmpdir + '/mpafiles.dgst'
    )
    ret, out = safe_syscall(cmd)
    assert ret == 0, 'Could not verify signature! %s' % out

    # verify digest file
    fd = open(tmpdir+'/mpafiles.dgst', 'r')

    file1_dgst = fd.readline()
    expected = ('SHA1(/example/path/file1)= '
                '1ff3a6a74eaa9b1b03a948fe9d3b3e327768af0c\n')
    assert file1_dgst == expected, (
        'Digest is incorrect: "%s" does not match "%s"' %
        (file1_dgst, expected)
    )

    expected = ('SHA1(/example/path/file2)= '
                '1ff3a6a74eaa9b1b03a948fe9d3b3e327768af0c\n')
    file2_dgst = fd.readline()
    assert file2_dgst == expected, \
        'Digest is incorrect: %s does not match %s' % (file2_dgst, expected)

    fd.close()

    # verify file path
    file1_path = tmpdir+conf[0]['path']
    assert os.path.isfile(file1_path), (
        'File does not exist at: %s' % file1_path
    )
    file2_path = tmpdir+conf[1]['path']
    assert os.path.isfile(file2_path), (
        'File does not exist at: %s' % file2_path
    )

    # verify the contents
    fd = open(tmpdir+'/example/path/file1', 'rb')
    file1_contents = fd.read()
    fd.close()
    file1_orig = conf[0]['contents'].encode('utf-8')
    assert file1_orig == file1_contents, (
        'Contents of file1 changed orig: %s now: %s!' %
        (file1_orig, file1_contents)
    )

    fd = open(tmpdir+'/example/path/file2', 'rb')
    file2_contents = fd.read()
    fd.close()
    file2_orig = conf[1]['contents'].encode('utf-8')
    assert file2_orig == file2_contents, (
        'Contents of file2 changed orig: %s now: %s!' %
        (file2_orig, file2_contents)
    )

    # Test MPA signing key generation
    mpa_signkey(expire_days=None, passphrase=None,
                mpasign_path='/tmp/mpasign')

    res, out = safe_syscall(
        ['ls', '-l', '/tmp/mpasign/mpasignature.pub'],
        raisemode=True
    )
    print(out)
    res, out = safe_syscall(
        ['ls', '-l', '/tmp/mpasign/mpasignkey.pem'],
        raisemode=True
    )
    print(out)

    pub = mpa_getsignpub(mpasign_path='/tmp/mpasign')
    assert '-----BEGIN PUBLIC KEY-----' in pub
    assert '-----END PUBLIC KEY-----' in pub

    key = mpa_getsignkey(mpasign_path='/tmp/mpasign')
    assert '-----BEGIN RSA PRIVATE KEY-----' in key
    assert '-----END RSA PRIVATE KEY-----' in key

    # Clean up.
    safe_syscall(['rm', '-rf', '/tmp/mpasign'])
    safe_syscall(['rm', '-rf', tmpdir])

    print('Tests successful!')


if __name__ == '__main__':
    import doctest
    doctest.testmod()
    test_all()
